/**
 * Wantuz Checkout for WooCommerce
 */
(function($) {
    'use strict';
    
    var WantuzCheckout = {
        
        init: function() {
            this.bindEvents();
            this.listenForMessages();
        },
        
        bindEvents: function() {
            var self = this;
            
            // Wantuz butonuna tıklama
            $(document).on('click', '#wantuz-checkout-btn', function(e) {
                e.preventDefault();
                self.openModal();
            });
            
            // Modal kapatma
            $(document).on('click', '.wantuz-modal-close', function(e) {
                e.preventDefault();
                self.closeModal();
            });
            
            // Modal dışına tıklama
            $(document).on('click', '#wantuz-modal', function(e) {
                if (e.target === this) {
                    self.closeModal();
                }
            });
            
            // ESC tuşu
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    self.closeModal();
                }
            });
        },
        
        openModal: function() {
            var iframe = $('#wantuz-iframe');
            iframe.attr('src', wantuzCheckout.widgetUrl);
            $('#wantuz-modal').fadeIn(200);
            $('body').css('overflow', 'hidden');
        },
        
        closeModal: function() {
            $('#wantuz-modal').fadeOut(200);
            $('#wantuz-iframe').attr('src', '');
            $('body').css('overflow', '');
        },
        
        listenForMessages: function() {
            var self = this;
            
            window.addEventListener('message', function(event) {
                // Güvenlik kontrolü
                if (event.data && event.data.type === 'wantuz-checkout-complete') {
                    self.handleCheckoutComplete(event.data);
                }
            });
        },
        
        handleCheckoutComplete: function(data) {
            var self = this;
            
            console.log('Wantuz Checkout Complete:', data);
            
            // Modal'ı kapat
            this.closeModal();
            
            // Formu doldur
            this.fillBillingForm(data.billingAddress);
            this.fillShippingForm(data.deliveryAddress);
            
            // Başarı mesajı göster
            this.showSuccessMessage();
            
            // Order ID'yi gizli alana kaydet (isteğe bağlı)
            if (!$('#wantuz_order_id').length) {
                $('<input>').attr({
                    type: 'hidden',
                    id: 'wantuz_order_id',
                    name: 'wantuz_order_id',
                    value: data.orderId
                }).appendTo('form.checkout');
            } else {
                $('#wantuz_order_id').val(data.orderId);
            }
        },
        
        fillBillingForm: function(address) {
            if (!address) return;
            
            // İsim soyisim ayrıştır
            var nameParts = this.parseFullName(address.fullName);
            
            // Fatura formu alanlarını doldur
            $('#billing_first_name').val(nameParts.firstName).trigger('change');
            $('#billing_last_name').val(nameParts.lastName).trigger('change');
            $('#billing_phone').val(this.formatPhone(address.phoneNumber)).trigger('change');
            $('#billing_country').val('TR').trigger('change');
            
            // Şehir ve ilçe
            setTimeout(function() {
                $('#billing_state').val(address.city).trigger('change');
                setTimeout(function() {
                    $('#billing_city').val(address.district).trigger('change');
                }, 300);
            }, 300);
            
            $('#billing_postcode').val(address.postalCode).trigger('change');
            
            // Adres satırı
            var addressLine = this.buildAddressLine(address);
            $('#billing_address_1').val(addressLine).trigger('change');
            
            // Ek bilgiler
            if (address.directions) {
                $('#billing_address_2').val(address.directions).trigger('change');
            }
            
            // Şirket bilgileri (fatura adresi ise)
            if (address.companyName) {
                $('#billing_company').val(address.companyName).trigger('change');
            }
        },
        
        fillShippingForm: function(address) {
            if (!address) return;
            
            // Farklı adrese gönder checkbox'ını işaretle
            if (!$('#ship-to-different-address-checkbox').is(':checked')) {
                $('#ship-to-different-address-checkbox').prop('checked', true).trigger('change');
            }
            
            // Biraz bekle, form görünür olsun
            setTimeout(function() {
                var nameParts = this.parseFullName(address.fullName);
                
                $('#shipping_first_name').val(nameParts.firstName).trigger('change');
                $('#shipping_last_name').val(nameParts.lastName).trigger('change');
                $('#shipping_country').val('TR').trigger('change');
                
                setTimeout(function() {
                    $('#shipping_state').val(address.city).trigger('change');
                    setTimeout(function() {
                        $('#shipping_city').val(address.district).trigger('change');
                    }, 300);
                }, 300);
                
                $('#shipping_postcode').val(address.postalCode).trigger('change');
                
                var addressLine = this.buildAddressLine(address);
                $('#shipping_address_1').val(addressLine).trigger('change');
                
                if (address.directions) {
                    $('#shipping_address_2').val(address.directions).trigger('change');
                }
            }.bind(this), 500);
        },
        
        parseFullName: function(fullName) {
            if (!fullName) return { firstName: '', lastName: '' };
            
            var parts = fullName.trim().split(' ');
            if (parts.length === 1) {
                return { firstName: parts[0], lastName: '' };
            }
            
            var lastName = parts.pop();
            var firstName = parts.join(' ');
            
            return { firstName: firstName, lastName: lastName };
        },
        
        formatPhone: function(phone) {
            if (!phone) return '';
            
            // Sadece rakamları al
            phone = phone.replace(/\D/g, '');
            
            // Başında 0 yoksa ekle
            if (phone.length === 10 && phone.charAt(0) !== '0') {
                phone = '0' + phone;
            }
            
            return phone;
        },
        
        buildAddressLine: function(address) {
            var parts = [];
            
            if (address.neighborhood) parts.push(address.neighborhood + ' Mah.');
            if (address.street) parts.push(address.street);
            if (address.avenue) parts.push(address.avenue);
            if (address.buildingName) parts.push(address.buildingName);
            if (address.buildingNumber) parts.push('No: ' + address.buildingNumber);
            if (address.floor) parts.push('Kat: ' + address.floor);
            if (address.doorNumber) parts.push('Daire: ' + address.doorNumber);
            
            return parts.join(' ');
        },
        
        showSuccessMessage: function() {
            var successHtml = 
                '<div class="wantuz-success">' +
                    '<div class="wantuz-success-icon">✓</div>' +
                    '<div class="wantuz-success-text">' +
                        '<p class="wantuz-success-title">Adresler başarıyla yüklendi!</p>' +
                        '<p class="wantuz-success-subtitle">Teslimat ve fatura bilgileriniz otomatik dolduruldu.</p>' +
                    '</div>' +
                '</div>';
            
            // Eski başarı mesajını kaldır
            $('.wantuz-success').remove();
            
            // Yeni mesajı ekle
            $('#wantuz-checkout-container').html(successHtml);
            
            // Formu güncelle
            $('body').trigger('update_checkout');
        }
    };
    
    // Sayfa yüklendiğinde başlat
    $(document).ready(function() {
        WantuzCheckout.init();
    });
    
})(jQuery);

