<?php
/**
 * Plugin Name: Wantuz Checkout
 * Plugin URI: https://wantuz.com/integrations/woocommerce
 * Description: Wantuz ile tek tıkla adres paylaşımı - Müşterileriniz kayıtlı adreslerini kolayca kullanabilir
 * Version: 1.0.0
 * Author: Wantuz
 * Author URI: https://wantuz.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wantuz-checkout
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 8.0
 */

if (!defined('ABSPATH')) {
    exit;
}

define('WANTUZ_CHECKOUT_VERSION', '1.0.0');
define('WANTUZ_CHECKOUT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WANTUZ_CHECKOUT_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Ana plugin sınıfı
 */
class Wantuz_Checkout {
    
    private static $instance = null;
    private $api_key = '';
    private $api_url = 'https://wantuz.com';
    
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function __construct() {
        $this->api_key = get_option('wantuz_api_key', '');
        
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        
        // WooCommerce hooks
        add_action('woocommerce_before_checkout_form', array($this, 'add_wantuz_button'), 5);
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // AJAX handlers
        add_action('wp_ajax_wantuz_get_address', array($this, 'ajax_get_address'));
        add_action('wp_ajax_nopriv_wantuz_get_address', array($this, 'ajax_get_address'));
        
        // REST API endpoints
        add_action('rest_api_init', array($this, 'register_rest_routes'));
    }
    
    /**
     * Admin menüsü ekle
     */
    public function add_admin_menu() {
        add_options_page(
            __('Wantuz Checkout Ayarları', 'wantuz-checkout'),
            __('Wantuz Checkout', 'wantuz-checkout'),
            'manage_options',
            'wantuz-checkout',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Ayarları kaydet
     */
    public function register_settings() {
        register_setting('wantuz_checkout_settings', 'wantuz_api_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        
        register_setting('wantuz_checkout_settings', 'wantuz_button_text', array(
            'type' => 'string',
            'default' => 'Wantuz ile Adres Kullan',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        
        register_setting('wantuz_checkout_settings', 'wantuz_button_position', array(
            'type' => 'string',
            'default' => 'before_form',
            'sanitize_callback' => 'sanitize_text_field',
        ));
    }
    
    /**
     * Ayarlar sayfası
     */
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div style="background:#fff;padding:20px;border-radius:8px;margin:20px 0;box-shadow:0 1px 3px rgba(0,0,0,0.1)">
                <div style="display:flex;align-items:center;gap:12px;margin-bottom:16px">
                    <img src="<?php echo WANTUZ_CHECKOUT_PLUGIN_URL; ?>assets/wantuz-logo.svg" alt="Wantuz" style="height:32px">
                    <div>
                        <h2 style="margin:0;font-size:18px">Wantuz Checkout</h2>
                        <p style="margin:4px 0 0;color:#666">Müşterileriniz kayıtlı adreslerini tek tıkla kullanabilir</p>
                    </div>
                </div>
                
                <p>
                    <strong>Nasıl Çalışır?</strong><br>
                    1. Müşteri "Wantuz ile Adres Kullan" butonuna tıklar<br>
                    2. Telefon numarası ile doğrulama yapar (SMS veya QR)<br>
                    3. Kayıtlı adreslerinden birini seçer<br>
                    4. Teslimat ve fatura bilgileri otomatik doldurulur
                </p>
            </div>
            
            <form method="post" action="options.php">
                <?php settings_fields('wantuz_checkout_settings'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="wantuz_api_key"><?php _e('API Anahtarı', 'wantuz-checkout'); ?></label>
                        </th>
                        <td>
                            <input type="text" 
                                   id="wantuz_api_key" 
                                   name="wantuz_api_key" 
                                   value="<?php echo esc_attr(get_option('wantuz_api_key')); ?>" 
                                   class="regular-text"
                                   placeholder="wantuz_xxxxx...">
                            <p class="description">
                                <?php _e('API anahtarı almak için', 'wantuz-checkout'); ?> 
                                <a href="mailto:api@wantuz.com">api@wantuz.com</a> 
                                <?php _e('adresine başvurun.', 'wantuz-checkout'); ?>
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="wantuz_button_text"><?php _e('Buton Metni', 'wantuz-checkout'); ?></label>
                        </th>
                        <td>
                            <input type="text" 
                                   id="wantuz_button_text" 
                                   name="wantuz_button_text" 
                                   value="<?php echo esc_attr(get_option('wantuz_button_text', 'Wantuz ile Adres Kullan')); ?>" 
                                   class="regular-text">
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="wantuz_button_position"><?php _e('Buton Konumu', 'wantuz-checkout'); ?></label>
                        </th>
                        <td>
                            <select id="wantuz_button_position" name="wantuz_button_position">
                                <option value="before_form" <?php selected(get_option('wantuz_button_position'), 'before_form'); ?>>
                                    <?php _e('Form öncesi', 'wantuz-checkout'); ?>
                                </option>
                                <option value="after_billing" <?php selected(get_option('wantuz_button_position'), 'after_billing'); ?>>
                                    <?php _e('Fatura adresi sonrası', 'wantuz-checkout'); ?>
                                </option>
                            </select>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(__('Ayarları Kaydet', 'wantuz-checkout')); ?>
            </form>
            
            <?php if (!empty($this->api_key)): ?>
            <div style="background:#f0fdf4;border:1px solid #86efac;padding:16px;border-radius:8px;margin-top:20px">
                <strong style="color:#166534">✓ API Anahtarı Yapılandırıldı</strong>
                <p style="margin:8px 0 0;color:#166534">Wantuz Checkout aktif ve çalışıyor.</p>
            </div>
            <?php else: ?>
            <div style="background:#fef2f2;border:1px solid #fecaca;padding:16px;border-radius:8px;margin-top:20px">
                <strong style="color:#991b1b">⚠ API Anahtarı Gerekli</strong>
                <p style="margin:8px 0 0;color:#991b1b">Wantuz Checkout'u kullanmak için API anahtarı girmeniz gerekmektedir.</p>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Checkout sayfasına Wantuz butonu ekle
     */
    public function add_wantuz_button() {
        if (empty($this->api_key)) {
            return;
        }
        
        $button_text = get_option('wantuz_button_text', 'Wantuz ile Adres Kullan');
        ?>
        <div id="wantuz-checkout-container" style="margin-bottom:24px">
            <button type="button" id="wantuz-checkout-btn" class="wantuz-btn">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"></path>
                    <circle cx="12" cy="10" r="3"></circle>
                </svg>
                <?php echo esc_html($button_text); ?>
            </button>
            <p class="wantuz-hint">Kayıtlı adreslerinizi tek tıkla kullanın</p>
        </div>
        
        <!-- Wantuz Modal -->
        <div id="wantuz-modal" class="wantuz-modal" style="display:none">
            <div class="wantuz-modal-content">
                <button type="button" class="wantuz-modal-close">&times;</button>
                <iframe id="wantuz-iframe" src="" frameborder="0"></iframe>
            </div>
        </div>
        <?php
    }
    
    /**
     * Script ve stilleri yükle
     */
    public function enqueue_scripts() {
        if (!is_checkout()) {
            return;
        }
        
        if (empty($this->api_key)) {
            return;
        }
        
        wp_enqueue_style(
            'wantuz-checkout',
            WANTUZ_CHECKOUT_PLUGIN_URL . 'assets/wantuz-checkout.css',
            array(),
            WANTUZ_CHECKOUT_VERSION
        );
        
        wp_enqueue_script(
            'wantuz-checkout',
            WANTUZ_CHECKOUT_PLUGIN_URL . 'assets/wantuz-checkout.js',
            array('jquery'),
            WANTUZ_CHECKOUT_VERSION,
            true
        );
        
        wp_localize_script('wantuz-checkout', 'wantuzCheckout', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wantuz_checkout'),
            'widgetUrl' => $this->api_url . '/checkout/widget?apiKey=' . $this->api_key,
            'i18n' => array(
                'loading' => __('Yükleniyor...', 'wantuz-checkout'),
                'error' => __('Bir hata oluştu', 'wantuz-checkout'),
            ),
        ));
    }
    
    /**
     * AJAX: Adres bilgilerini al
     */
    public function ajax_get_address() {
        check_ajax_referer('wantuz_checkout', 'nonce');
        
        $order_id = sanitize_text_field($_POST['order_id'] ?? '');
        
        if (empty($order_id)) {
            wp_send_json_error('Order ID gerekli');
        }
        
        $response = wp_remote_get($this->api_url . '/merchant/orders/' . $order_id, array(
            'headers' => array(
                'x-api-key' => $this->api_key,
                'Content-Type' => 'application/json',
            ),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (wp_remote_retrieve_response_code($response) !== 200) {
            wp_send_json_error($body['message'] ?? 'API hatası');
        }
        
        wp_send_json_success($body);
    }
    
    /**
     * REST API route'larını kaydet
     */
    public function register_rest_routes() {
        register_rest_route('wantuz/v1', '/callback', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_callback'),
            'permission_callback' => '__return_true',
        ));
    }
    
    /**
     * Wantuz callback handler
     */
    public function handle_callback($request) {
        $data = $request->get_json_params();
        
        // Callback işlemleri burada yapılabilir
        // Örneğin: Order meta'ya kaydetme, loglama vb.
        
        return new WP_REST_Response(array('success' => true), 200);
    }
}

/**
 * Plugin'i başlat
 */
function wantuz_checkout_init() {
    // WooCommerce aktif mi kontrol et
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function() {
            ?>
            <div class="notice notice-error">
                <p><?php _e('Wantuz Checkout eklentisi WooCommerce gerektirir. Lütfen WooCommerce\'i yükleyin ve etkinleştirin.', 'wantuz-checkout'); ?></p>
            </div>
            <?php
        });
        return;
    }
    
    return Wantuz_Checkout::instance();
}
add_action('plugins_loaded', 'wantuz_checkout_init');

/**
 * Eklenti aktivasyon hook'u
 */
register_activation_hook(__FILE__, function() {
    // Varsayılan ayarları kaydet
    add_option('wantuz_button_text', 'Wantuz ile Adres Kullan');
    add_option('wantuz_button_position', 'before_form');
});

/**
 * Eklenti deaktivasyon hook'u
 */
register_deactivation_hook(__FILE__, function() {
    // Gerekirse temizlik işlemleri
});

