/**
 * Wantuz Checkout for Shopify
 * 
 * Kullanım:
 * <script src="https://wantuz.com/integrations/shopify/wantuz-checkout.js?apiKey=YOUR_API_KEY"></script>
 */
(function() {
  'use strict';
  
  // URL'den API key'i al
  var scripts = document.getElementsByTagName('script');
  var currentScript = scripts[scripts.length - 1];
  var scriptSrc = currentScript.src;
  var apiKeyMatch = scriptSrc.match(/[?&]apiKey=([^&]+)/);
  
  if (!apiKeyMatch) {
    console.error('[Wantuz] API key not found in script URL');
    return;
  }
  
  var CONFIG = {
    apiKey: apiKeyMatch[1],
    apiUrl: 'https://wantuz.com',
    buttonText: 'Wantuz ile Adres Kullan'
  };
  
  // Stiller
  var styles = '\
    .wantuz-checkout-wrapper {\
      margin: 16px 0;\
      padding: 16px;\
      background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);\
      border: 1px solid #e2e8f0;\
      border-radius: 12px;\
    }\
    .wantuz-btn {\
      display: flex;\
      align-items: center;\
      justify-content: center;\
      gap: 10px;\
      width: 100%;\
      padding: 14px 20px;\
      background: linear-gradient(135deg, #0f172a 0%, #1e293b 100%);\
      color: #fff;\
      border: none;\
      border-radius: 10px;\
      font-size: 15px;\
      font-weight: 600;\
      cursor: pointer;\
      transition: all 0.2s;\
      box-shadow: 0 4px 12px rgba(15, 23, 42, 0.2);\
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;\
    }\
    .wantuz-btn:hover {\
      transform: translateY(-1px);\
      box-shadow: 0 6px 16px rgba(15, 23, 42, 0.3);\
    }\
    .wantuz-hint {\
      text-align: center;\
      margin: 8px 0 0;\
      font-size: 12px;\
      color: #64748b;\
    }\
    .wantuz-modal {\
      display: none;\
      position: fixed;\
      top: 0;\
      left: 0;\
      width: 100%;\
      height: 100%;\
      background: rgba(0, 0, 0, 0.6);\
      z-index: 999999;\
      align-items: center;\
      justify-content: center;\
      padding: 20px;\
      box-sizing: border-box;\
    }\
    .wantuz-modal.active {\
      display: flex;\
    }\
    .wantuz-modal-content {\
      position: relative;\
      width: 100%;\
      max-width: 460px;\
      height: 85vh;\
      max-height: 680px;\
      background: #fff;\
      border-radius: 16px;\
      overflow: hidden;\
      box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);\
    }\
    .wantuz-modal-close {\
      position: absolute;\
      top: 10px;\
      right: 10px;\
      width: 32px;\
      height: 32px;\
      background: rgba(255,255,255,0.95);\
      border: none;\
      border-radius: 50%;\
      font-size: 20px;\
      cursor: pointer;\
      z-index: 10;\
      display: flex;\
      align-items: center;\
      justify-content: center;\
      color: #475569;\
      box-shadow: 0 2px 8px rgba(0,0,0,0.1);\
    }\
    .wantuz-iframe {\
      width: 100%;\
      height: 100%;\
      border: none;\
    }\
    .wantuz-success {\
      display: flex;\
      align-items: center;\
      gap: 12px;\
      padding: 14px;\
      background: #f0fdf4;\
      border: 1px solid #86efac;\
      border-radius: 10px;\
    }\
    .wantuz-success-icon {\
      width: 36px;\
      height: 36px;\
      background: #22c55e;\
      border-radius: 50%;\
      display: flex;\
      align-items: center;\
      justify-content: center;\
      color: #fff;\
      font-size: 18px;\
      flex-shrink: 0;\
    }\
    .wantuz-success-text h4 {\
      margin: 0 0 2px;\
      font-size: 14px;\
      font-weight: 600;\
      color: #166534;\
    }\
    .wantuz-success-text p {\
      margin: 0;\
      font-size: 12px;\
      color: #15803d;\
    }\
    @media (max-width: 480px) {\
      .wantuz-modal-content {\
        height: 100vh;\
        max-height: none;\
        border-radius: 0;\
      }\
      .wantuz-modal {\
        padding: 0;\
      }\
    }\
  ';
  
  // Stilleri ekle
  var styleEl = document.createElement('style');
  styleEl.textContent = styles;
  document.head.appendChild(styleEl);
  
  // DOM hazır olduğunda başlat
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
  
  function init() {
    // Checkout sayfasında mıyız kontrol et
    if (!isCheckoutPage()) {
      return;
    }
    
    // Checkout formunu bul
    var checkoutForm = findCheckoutForm();
    
    if (!checkoutForm) {
      // Biraz bekleyip tekrar dene
      setTimeout(function() {
        checkoutForm = findCheckoutForm();
        if (checkoutForm) {
          injectButton(checkoutForm);
        }
      }, 1500);
      return;
    }
    
    injectButton(checkoutForm);
  }
  
  function isCheckoutPage() {
    return window.location.pathname.includes('/checkout') ||
           window.location.pathname.includes('/cart') ||
           document.querySelector('[data-customer-information-form]') !== null;
  }
  
  function findCheckoutForm() {
    return document.querySelector('[data-customer-information-form]') ||
           document.querySelector('.section--shipping-address') ||
           document.querySelector('[data-address-fields]') ||
           document.querySelector('.step__sections');
  }
  
  function injectButton(form) {
    if (document.getElementById('wantuz-checkout-wrapper')) return;
    
    // Wrapper
    var wrapper = document.createElement('div');
    wrapper.id = 'wantuz-checkout-wrapper';
    wrapper.className = 'wantuz-checkout-wrapper';
    wrapper.innerHTML = 
      '<button type="button" id="wantuz-btn" class="wantuz-btn">' +
        '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">' +
          '<path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"></path>' +
          '<circle cx="12" cy="10" r="3"></circle>' +
        '</svg>' +
        CONFIG.buttonText +
      '</button>' +
      '<p class="wantuz-hint">Kayıtlı adreslerinizi tek tıkla kullanın</p>';
    
    // Modal
    var modal = document.createElement('div');
    modal.id = 'wantuz-modal';
    modal.className = 'wantuz-modal';
    modal.innerHTML = 
      '<div class="wantuz-modal-content">' +
        '<button type="button" class="wantuz-modal-close">&times;</button>' +
        '<iframe class="wantuz-iframe" id="wantuz-iframe" src=""></iframe>' +
      '</div>';
    
    form.insertBefore(wrapper, form.firstChild);
    document.body.appendChild(modal);
    
    // Events
    document.getElementById('wantuz-btn').addEventListener('click', openModal);
    modal.querySelector('.wantuz-modal-close').addEventListener('click', closeModal);
    modal.addEventListener('click', function(e) {
      if (e.target === modal) closeModal();
    });
    document.addEventListener('keydown', function(e) {
      if (e.key === 'Escape') closeModal();
    });
    
    window.addEventListener('message', handleMessage);
  }
  
  function openModal() {
    var modal = document.getElementById('wantuz-modal');
    var iframe = document.getElementById('wantuz-iframe');
    iframe.src = CONFIG.apiUrl + '/checkout/widget?apiKey=' + CONFIG.apiKey;
    modal.classList.add('active');
    document.body.style.overflow = 'hidden';
  }
  
  function closeModal() {
    var modal = document.getElementById('wantuz-modal');
    var iframe = document.getElementById('wantuz-iframe');
    if (modal) {
      modal.classList.remove('active');
      iframe.src = '';
      document.body.style.overflow = '';
    }
  }
  
  function handleMessage(event) {
    if (event.data && event.data.type === 'wantuz-checkout-complete') {
      closeModal();
      fillForm(event.data);
      showSuccess();
    }
  }
  
  function fillForm(data) {
    var shipping = data.deliveryAddress;
    var billing = data.billingAddress;
    
    if (shipping) {
      var nameParts = parseFullName(shipping.fullName);
      
      setField('[name="checkout[shipping_address][first_name]"]', nameParts.firstName);
      setField('[name="checkout[shipping_address][last_name]"]', nameParts.lastName);
      setField('[name="checkout[shipping_address][phone]"]', formatPhone(shipping.phoneNumber));
      setField('[name="checkout[shipping_address][address1]"]', buildAddressLine(shipping));
      setField('[name="checkout[shipping_address][address2]"]', shipping.directions || '');
      setField('[name="checkout[shipping_address][city]"]', shipping.district);
      setField('[name="checkout[shipping_address][zip]"]', shipping.postalCode);
      
      setSelectByText('[name="checkout[shipping_address][province]"]', shipping.city);
    }
    
    if (billing && billing.id !== (shipping && shipping.id)) {
      var differentBilling = document.querySelector('[name="checkout[different_billing_address]"]');
      if (differentBilling && !differentBilling.checked) {
        differentBilling.click();
      }
      
      setTimeout(function() {
        var nameParts = parseFullName(billing.fullName);
        
        setField('[name="checkout[billing_address][first_name]"]', nameParts.firstName);
        setField('[name="checkout[billing_address][last_name]"]', nameParts.lastName);
        setField('[name="checkout[billing_address][phone]"]', formatPhone(billing.phoneNumber));
        setField('[name="checkout[billing_address][address1]"]', buildAddressLine(billing));
        setField('[name="checkout[billing_address][city]"]', billing.district);
        setField('[name="checkout[billing_address][zip]"]', billing.postalCode);
        
        if (billing.companyName) {
          setField('[name="checkout[billing_address][company]"]', billing.companyName);
        }
      }, 500);
    }
  }
  
  function setField(selector, value) {
    var field = document.querySelector(selector);
    if (field) {
      field.value = value;
      field.dispatchEvent(new Event('input', { bubbles: true }));
      field.dispatchEvent(new Event('change', { bubbles: true }));
    }
  }
  
  function setSelectByText(selector, text) {
    var select = document.querySelector(selector);
    if (select) {
      for (var i = 0; i < select.options.length; i++) {
        if (select.options[i].text.toLowerCase().includes(text.toLowerCase())) {
          select.selectedIndex = i;
          select.dispatchEvent(new Event('change', { bubbles: true }));
          break;
        }
      }
    }
  }
  
  function parseFullName(name) {
    if (!name) return { firstName: '', lastName: '' };
    var parts = name.trim().split(' ');
    if (parts.length === 1) return { firstName: parts[0], lastName: '' };
    var lastName = parts.pop();
    return { firstName: parts.join(' '), lastName: lastName };
  }
  
  function formatPhone(phone) {
    if (!phone) return '';
    phone = phone.replace(/\D/g, '');
    if (phone.length === 10 && phone.charAt(0) !== '0') {
      phone = '0' + phone;
    }
    return phone;
  }
  
  function buildAddressLine(addr) {
    var parts = [];
    if (addr.neighborhood) parts.push(addr.neighborhood + ' Mah.');
    if (addr.street) parts.push(addr.street);
    if (addr.buildingNumber) parts.push('No: ' + addr.buildingNumber);
    if (addr.floor) parts.push('Kat: ' + addr.floor);
    if (addr.doorNumber) parts.push('Daire: ' + addr.doorNumber);
    return parts.join(' ');
  }
  
  function showSuccess() {
    var wrapper = document.getElementById('wantuz-checkout-wrapper');
    if (wrapper) {
      wrapper.innerHTML = 
        '<div class="wantuz-success">' +
          '<div class="wantuz-success-icon">✓</div>' +
          '<div class="wantuz-success-text">' +
            '<h4>Adresler yüklendi!</h4>' +
            '<p>Teslimat ve fatura bilgileri dolduruldu.</p>' +
          '</div>' +
        '</div>';
    }
  }
})();

